import { Component, ElementRef, EventEmitter, HostBinding, HostListener, Inject, Input, Output, ViewChild } from '@angular/core';
import { DOCUMENT } from '@angular/common';
import { animate, state, style, transition, trigger } from '@angular/animations';
import { coerceBooleanProperty } from '@angular/cdk/coercion';
import { ModalContentComponent } from '../modal-content/modal-content.component';
import * as i0 from "@angular/core";
import * as i1 from "../modal.service";
import * as i2 from "../../backdrop/backdrop.service";
import * as i3 from "@angular/cdk/a11y";
import * as i4 from "../modal-content/modal-content.component";
import * as i5 from "../modal-dialog/modal-dialog.component";
export class ModalComponent {
    constructor(document, renderer, hostElement, modalService, backdropService) {
        this.document = document;
        this.renderer = renderer;
        this.hostElement = hostElement;
        this.modalService = modalService;
        this.backdropService = backdropService;
        /**
         * Align the modal in the center or top of the screen.
         * @type {'top' | 'center'}
         * @default 'top'
         */
        this.alignment = 'top';
        /**
         * Apply a backdrop on body while modal is open.
         * @type boolean | 'static'
         * @default true
         */
        this.backdrop = true;
        /**
         * Closes the modal when escape key is pressed.
         * @type boolean
         * @default true
         */
        this.keyboard = true;
        /**
         * Remove animation to create modal that simply appear rather than fade in to view.
         */
        this.transition = true;
        /**
         * Default role for modal. [docs]
         * @type string
         * @default 'dialog'
         */
        this.role = 'dialog';
        /**
         * Set aria-modal html attr for modal. [docs]
         * @type boolean
         * @default true
         */
        this.ariaModal = true;
        this._scrollable = false;
        /**
         * Event triggered on modal dismiss.
         */
        this.visibleChange = new EventEmitter();
        this._show = true;
        this.mouseDownTarget = null;
    }
    /**
     * Create a scrollable modal that allows scrolling the modal body.
     * @type boolean
     */
    set scrollable(value) {
        this._scrollable = coerceBooleanProperty(value);
    }
    get scrollable() {
        return this._scrollable;
    }
    /**
     * Toggle the visibility of modal component.
     * @type boolean
     */
    set visible(value) {
        const newValue = coerceBooleanProperty(value);
        if (this._visible !== newValue) {
            this._visible = newValue;
            this.setBackdrop(this.backdrop !== false && newValue);
            this.setBodyStyles(newValue);
            this.visibleChange.emit(newValue);
        }
    }
    get visible() {
        return this._visible;
    }
    get hostClasses() {
        return {
            modal: true,
            fade: this.transition,
            show: this.show,
        };
    }
    get ariaHidden() {
        return this.visible ? null : true;
    }
    ;
    get tabIndex() {
        return '-1';
    }
    get animateTrigger() {
        return this.visible ? 'visible' : 'hidden';
    }
    get show() {
        return this.visible && this._show;
    }
    set show(value) {
        this._show = value;
    }
    get scrollbarWidth() {
        // https://developer.mozilla.org/en-US/docs/Web/API/Window/innerWidth#usage_notes
        const documentWidth = this.document.documentElement.clientWidth;
        const scrollbarWidth = Math.abs((window?.innerWidth ?? documentWidth) - documentWidth);
        return `${scrollbarWidth}px`;
    }
    animateStart(event) {
        this.renderer.setStyle(this.hostElement.nativeElement, 'overflow-y', 'hidden');
        this.show = this.visible;
    }
    animateDone(event) {
        const scrollbarWidth = this.scrollbarWidth;
        setTimeout(() => {
            if (event.toState === 'visible') {
                this.inBoundingClientRect = this.modalContent?.nativeElement?.getBoundingClientRect().height <= this.document?.documentElement.clientHeight;
                if (!this.inBoundingClientRect) {
                    this.renderer.setStyle(this.document.body, 'overflow-y', 'hidden');
                    this.renderer.setStyle(this.document.body, 'padding-right', scrollbarWidth);
                }
                else {
                    this.renderer.removeStyle(this.document.body, 'overflow-y');
                    this.renderer.removeStyle(this.document.body, 'padding-right');
                }
                if (this.fullscreen === true) {
                    this.renderer.setStyle(this.document.body, 'overflow-y', 'hidden');
                }
                this.renderer.removeStyle(this.hostElement.nativeElement, 'overflow-y');
            }
        });
    }
    onKeyDownHandler(event) {
        if (event.key === 'Escape' && this.keyboard && this.visible) {
            if (this.backdrop === 'static') {
                this.setStaticBackdrop();
            }
            else {
                this.modalService.toggle({ show: false, modal: this });
            }
        }
    }
    onMouseDownHandler($event) {
        this.mouseDownTarget = $event.target;
    }
    onClickHandler($event) {
        $event.stopPropagation();
        if (this.mouseDownTarget !== $event.target) {
            this.mouseDownTarget = null;
            return;
        }
        if (this.backdrop === 'static') {
            this.setStaticBackdrop();
            return;
        }
        this.modalService.toggle({ show: false, modal: this });
    }
    ngOnInit() {
        this.stateToggleSubscribe();
    }
    ngOnDestroy() {
        this.modalService.toggle({ show: false, modal: this });
        this.stateToggleSubscribe(false);
    }
    stateToggleSubscribe(subscribe = true) {
        if (subscribe) {
            this.stateToggleSubscription = this.modalService.modalState$.subscribe((action) => {
                if (this === action.modal || this.id === action.id) {
                    if ('show' in action) {
                        this.visible = action?.show === 'toggle' ? !this.visible : action.show;
                    }
                }
                else {
                    if (this.visible) {
                        this.visible = false;
                    }
                }
            });
        }
        else {
            this.stateToggleSubscription.unsubscribe();
        }
    }
    setBackdrop(setBackdrop) {
        if (setBackdrop) {
            this.activeBackdrop = this.backdropService.setBackdrop('modal');
        }
        else {
            this.activeBackdrop = this.backdropService.clearBackdrop(this.activeBackdrop);
        }
    }
    setBodyStyles(open) {
        if (open) {
            if (this.fullscreen === true) {
                this.renderer.setStyle(this.document.body, 'overflow-y', 'hidden');
            }
            if (this.backdrop === true) {
                this.renderer.addClass(this.document.body, 'modal-open');
            }
        }
        else {
            this.renderer.removeClass(this.document.body, 'modal-open');
            this.renderer.removeStyle(this.document.body, 'overflow-y');
            this.renderer.removeStyle(this.document.body, 'padding-right');
        }
    }
    setStaticBackdrop() {
        if (this.transition) {
            this.renderer.addClass(this.hostElement.nativeElement, 'modal-static');
            this.renderer.setStyle(this.hostElement.nativeElement, 'overflow-y', 'hidden');
            setTimeout(() => {
                this.renderer.removeClass(this.hostElement.nativeElement, 'modal-static');
                this.renderer.removeStyle(this.hostElement.nativeElement, 'overflow-y');
            }, 400);
        }
    }
}
ModalComponent.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.0.5", ngImport: i0, type: ModalComponent, deps: [{ token: DOCUMENT }, { token: i0.Renderer2 }, { token: i0.ElementRef }, { token: i1.ModalService }, { token: i2.BackdropService }], target: i0.ɵɵFactoryTarget.Component });
ModalComponent.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "14.0.5", type: ModalComponent, selector: "c-modal", inputs: { alignment: "alignment", backdrop: "backdrop", fullscreen: "fullscreen", keyboard: "keyboard", id: "id", size: "size", transition: "transition", role: "role", ariaModal: "ariaModal", scrollable: "scrollable", visible: "visible" }, outputs: { visibleChange: "visibleChange" }, host: { listeners: { "@showHide.start": "animateStart($event)", "@showHide.done": "animateDone($event)", "document:keydown": "onKeyDownHandler($event)", "mousedown": "onMouseDownHandler($event)", "click": "onClickHandler($event)" }, properties: { "attr.role": "this.role", "attr.aria-modal": "this.ariaModal", "class": "this.hostClasses", "attr.aria-hidden": "this.ariaHidden", "attr.tabindex": "this.tabIndex", "@showHide": "this.animateTrigger" } }, viewQueries: [{ propertyName: "modalContent", first: true, predicate: ModalContentComponent, descendants: true, read: ElementRef }], exportAs: ["cModal"], ngImport: i0, template: "<c-modal-dialog\n  [alignment]=\"alignment\"\n  [fullscreen]=\"fullscreen\"\n  [scrollable]=\"scrollable\"\n  [size]=\"size\">\n  <c-modal-content>\n    <div [cdkTrapFocus]=\"visible\" [cdkTrapFocusAutoCapture]=\"visible\" style=\"display: contents;\">\n      <ng-content></ng-content>\n    </div>\n  </c-modal-content>\n</c-modal-dialog>\n", styles: [""], dependencies: [{ kind: "directive", type: i3.CdkTrapFocus, selector: "[cdkTrapFocus]", inputs: ["cdkTrapFocus", "cdkTrapFocusAutoCapture"], exportAs: ["cdkTrapFocus"] }, { kind: "component", type: i4.ModalContentComponent, selector: "c-modal-content" }, { kind: "component", type: i5.ModalDialogComponent, selector: "c-modal-dialog", inputs: ["alignment", "fullscreen", "scrollable", "size"] }], animations: [
        trigger('showHide', [
            state('visible', style({
                display: 'block'
            })),
            state('hidden', style({
                display: 'none'
            })),
            transition('visible <=> *', [animate('300ms')])
        ])
    ] });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.0.5", ngImport: i0, type: ModalComponent, decorators: [{
            type: Component,
            args: [{ selector: 'c-modal', animations: [
                        trigger('showHide', [
                            state('visible', style({
                                display: 'block'
                            })),
                            state('hidden', style({
                                display: 'none'
                            })),
                            transition('visible <=> *', [animate('300ms')])
                        ])
                    ], exportAs: 'cModal', template: "<c-modal-dialog\n  [alignment]=\"alignment\"\n  [fullscreen]=\"fullscreen\"\n  [scrollable]=\"scrollable\"\n  [size]=\"size\">\n  <c-modal-content>\n    <div [cdkTrapFocus]=\"visible\" [cdkTrapFocusAutoCapture]=\"visible\" style=\"display: contents;\">\n      <ng-content></ng-content>\n    </div>\n  </c-modal-content>\n</c-modal-dialog>\n" }]
        }], ctorParameters: function () { return [{ type: undefined, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i0.Renderer2 }, { type: i0.ElementRef }, { type: i1.ModalService }, { type: i2.BackdropService }]; }, propDecorators: { alignment: [{
                type: Input
            }], backdrop: [{
                type: Input
            }], fullscreen: [{
                type: Input
            }], keyboard: [{
                type: Input
            }], id: [{
                type: Input
            }], size: [{
                type: Input
            }], transition: [{
                type: Input
            }], role: [{
                type: Input
            }, {
                type: HostBinding,
                args: ['attr.role']
            }], ariaModal: [{
                type: Input
            }, {
                type: HostBinding,
                args: ['attr.aria-modal']
            }], scrollable: [{
                type: Input
            }], visible: [{
                type: Input
            }], visibleChange: [{
                type: Output
            }], modalContent: [{
                type: ViewChild,
                args: [ModalContentComponent, { read: ElementRef }]
            }], hostClasses: [{
                type: HostBinding,
                args: ['class']
            }], ariaHidden: [{
                type: HostBinding,
                args: ['attr.aria-hidden']
            }], tabIndex: [{
                type: HostBinding,
                args: ['attr.tabindex']
            }], animateTrigger: [{
                type: HostBinding,
                args: ['@showHide']
            }], animateStart: [{
                type: HostListener,
                args: ['@showHide.start', ['$event']]
            }], animateDone: [{
                type: HostListener,
                args: ['@showHide.done', ['$event']]
            }], onKeyDownHandler: [{
                type: HostListener,
                args: ['document:keydown', ['$event']]
            }], onMouseDownHandler: [{
                type: HostListener,
                args: ['mousedown', ['$event']]
            }], onClickHandler: [{
                type: HostListener,
                args: ['click', ['$event']]
            }] } });
//# sourceMappingURL=data:application/json;base64,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