<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agent;
use App\Models\Bank;
use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Barryvdh\DomPDF\Facade\Pdf;   // 👈 PDF Package

class AgentController extends Controller
{
    public function index()
    {
        $agents = Agent::with(['bank', 'branch'])->paginate(10);
        return view('admin.agent.index', compact('agents'));
    }

    public function create()
    {
        $banks = Bank::active()->get();
        return view('admin.agent.create', compact('banks'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'branch_id' => 'required|exists:branches,id',
            'designation' => 'required|string|max:100',
            'name' => 'required|string|max:150',
            'email' => 'required|email|unique:agents,email',
            'mobile_number' => 'required|digits:10|unique:agents,mobile_number',
            'whatsapp_number' => 'nullable|digits:10',
            'password' => 'required|min:6',
            'image' => 'nullable|image|max:2048',
        ]);

        $data = $request->except(['password', 'image']);
        $data['password'] = Hash::make($request->password);

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('agents', 'public');
        }

        Agent::create($data);

        return redirect()->route('admin.agent.index')->with('success', 'Agent created successfully!');
    }

    public function edit(Agent $agent)
    {
        $banks = Bank::active()->get();
        $branches = Branch::where('bank_id', $agent->bank_id)->get();
        return view('admin.agent.edit', compact('agent', 'banks', 'branches'));
    }

    public function update(Request $request, Agent $agent)
    {
        $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'branch_id' => 'required|exists:branches,id',
            'designation' => 'required|string|max:100',
            'name' => 'required|string|max:150',
            'email' => 'required|email|unique:agents,email,' . $agent->id,
            'mobile_number' => 'required|digits:10|unique:agents,mobile_number,' . $agent->id,
            'whatsapp_number' => 'nullable|digits:10',
            'password' => 'nullable|min:6',
            'image' => 'nullable|image|max:2048',
        ]);

        $data = $request->except(['password', 'image']);
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }
        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('agents', 'public');
        }

        $agent->update($data);

        return redirect()->route('admin.agent.index')->with('success', 'Agent updated successfully!');
    }

    public function destroy(Agent $agent)
    {
        $agent->delete();
        return redirect()->route('admin.agent.index')->with('success', 'Agent deleted successfully!');
    }

    // 📌 New method for PDF Certificate
    public function certificate($id)
    {
        $agent = Agent::with(['bank', 'branch'])->findOrFail($id);

        $data = [
            'agent' => $agent,
            'date' => now()->format('d/m/Y'),
        ];

        $pdf = Pdf::loadView('admin.agent.certificate', $data)
            ->setPaper('a4', 'portrait');

        return $pdf->download('certificate_' . $agent->name . '.pdf');
    }
    // app/Http/Controllers/Admin/AgentController.php
    public function invoiceIndex()
    {
        $agents = Agent::paginate(10); // Example: all agents for invoice
        return view('admin.invoices.index', compact('agents'));
    }

}
